<?php
namespace PhpSigep\Services\Real;

use PhpSigep\Model\Destinatario;
use PhpSigep\Model\Destino;
use PhpSigep\Model\DestinoInternacional;
use PhpSigep\Model\DestinoNacional;
use PhpSigep\Model\Dimensao;
use PhpSigep\Model\FechaPlpVariosServicosRetorno;
use PhpSigep\Model\ObjetoPostal;
use PhpSigep\Model\PreListaDePostagem;
use PhpSigep\Model\ServicoAdicional;
use PhpSigep\Services\Result;
use PhpSigep\Model\ServicoDePostagem;

/**
 * @author: Stavarengo
 */
class FecharPreListaDePostagem
{

    /**
     * @param PreListaDePostagem $params
     *
     * @return \PhpSigep\Services\Result<\PhpSigep\Model\FechaPlpVariosServicosRetorno>
     */
    public function execute(\PhpSigep\Model\PreListaDePostagem $params)
    {
        $xmlDaPreLista = $this->getPlpXml($params);

        $listaEtiquetas = array();
        foreach ($params->getEncomendas() as $objetoPostal) {
            $listaEtiquetas[] = $objetoPostal->getEtiqueta()->getEtiquetaSemDv();
        }

        $xml = utf8_encode($xmlDaPreLista->flush());
//		$xml = utf8_encode($xml);
//		$xml = iconv('UTF-8', 'ISO-8859-1', $xml);

        $soapArgs = array(
            'xml'            => $xml,
            'idPlpCliente'   => '',
            'cartaoPostagem' => $params->getAccessData()->getCartaoPostagem(),
            'listaEtiquetas' => $listaEtiquetas,
            'usuario'        => $params->getAccessData()->getUsuario(),
            'senha'          => $params->getAccessData()->getSenha(),
        );
        
        $result = new Result();
        try {
            $r = SoapClientFactory::getSoapClient()->fechaPlpVariosServicos($soapArgs);
            if (class_exists('\StaLib_Logger',false)) {
                \StaLib_Logger::log('Retorno SIGEP fecha PLP: ' . print_r($r, true));
            }
            if ($r instanceof \SoapFault) {
                throw $r;
            }
            if ($r && $r->return) {
                $result->setResult(new FechaPlpVariosServicosRetorno(array('idPlp' => $r->return,'XMLPlp'=>$xml)));
            } else {
                $result->setErrorCode(0);
                $result->setErrorMsg('A resposta do Correios não está no formato esperado. Resposta recebida: "' . 
                    $r . '"');
            }
        } catch (\Exception $e) {
            if ($e instanceof \SoapFault) {
                $result->setIsSoapFault(true);
                $result->setErrorCode($e->getCode());
                $result->setErrorMsg(SoapClientFactory::convertEncoding($e->getMessage()));
            } else {
                $result->setErrorCode($e->getCode());
                $result->setErrorMsg($e->getMessage());
            }
        }
        
        return $result;
    }

    private function getPlpXml(PreListaDePostagem $data)
    {
        $writer = new \XMLWriter();
        $writer->openMemory();
        $writer->setIndentString("");
        $writer->setIndent(false);
        $writer->startDocument('1.0', 'UTF-8');

        $writer->startElement('correioslog');
        $writer->writeElement('tipo_arquivo', 'Postagem');
        $writer->writeElement('versao_arquivo', '2.3');
        $this->writePlp($writer, $data);
        $this->writeRemetente($writer, $data);
        $this->writeFormaPagamento($writer, $data);
        foreach ($data->getEncomendas() as $objetoPostal) {
            $this->writeObjetoPostal($writer, $objetoPostal);
        }
        $writer->endElement();

        return $writer;
    }

    private function writePlp(\XMLWriter $writer, PreListaDePostagem $data)
    {
        $writer->startElement('plp');
        $writer->writeElement('id_plp');
        $writer->writeElement('valor_global');
        $writer->writeElement('mcu_unidade_postagem');
        $writer->writeElement('nome_unidade_postagem');
        $writer->writeElement('cartao_postagem', $data->getAccessData()->getCartaoPostagem());
        $writer->endElement();
    }

    private function writeRemetente(\XMLWriter $writer, PreListaDePostagem $data)
    {
        $writer->startElement('remetente');
        $writer->writeElement('numero_contrato', $data->getAccessData()->getNumeroContrato());
        $writer->writeElement('numero_diretoria', $data->getAccessData()->getDiretoria()->getNumero());
        $writer->writeElement('codigo_administrativo', $data->getAccessData()->getCodAdministrativo());
        $writer->startElement('nome_remetente');
        $writer->writeCData($this->_($data->getRemetente()->getNome(), 50));
        $writer->endElement();
        $writer->startElement('logradouro_remetente');
        $writer->writeCdata($this->_($data->getRemetente()->getLogradouro(), 40));
        $writer->endElement();
        $writer->startElement('numero_remetente');
        $numero_remetente = $data->getRemetente()->getNumero();
        $writer->writeCdata($this->_(($numero_remetente ? $numero_remetente : 's/n'), 6));
        $writer->endElement();
        $writer->startElement('complemento_remetente');
        $writer->writeCdata($this->_($data->getRemetente()->getComplemento(), 20));
        $writer->endElement();
        $writer->startElement('bairro_remetente');
        $writer->writeCdata($this->_($data->getRemetente()->getBairro(), 20));
        $writer->endElement();
        $writer->startElement('cep_remetente');
        $writer->writeCdata($this->_(preg_replace('/[^\d]/', '', $data->getRemetente()->getCep()), 8));
        $writer->endElement();
        $writer->startElement('cidade_remetente');
        $writer->writeCdata($this->_($data->getRemetente()->getCidade(), 30));
        $writer->endElement();
        $writer->writeElement('uf_remetente', $this->_($data->getRemetente()->getUf(), 2, false));
        $writer->startElement('telefone_remetente');
        $writer->writeCdata($this->_(preg_replace('/[^\d]/', '', $data->getRemetente()->getTelefone()), 12));
        $writer->endElement();
        $writer->startElement('fax_remetente');
        $writer->writeCdata($this->_(preg_replace('/[^\d]/', '', $data->getRemetente()->getFax()), 12));
        $writer->endElement();
        $writer->startElement('email_remetente');
        $writer->writeCdata($this->_($data->getRemetente()->getEmail(), 50));
        $writer->endElement();
        $writer->endElement();
    }

    private function writeFormaPagamento(\XMLWriter $writer, PreListaDePostagem $data)
    {
        $writer->writeElement('forma_pagamento');
    }

    private function writeObjetoPostal(\XMLWriter $writer, ObjetoPostal $objetoPostal)
    {
        $writer->startElement('objeto_postal');
        $writer->writeElement('numero_etiqueta', $objetoPostal->getEtiqueta()->getEtiquetaComDv());
        $writer->writeElement('codigo_objeto_cliente');
        $writer->writeElement('codigo_servico_postagem', $objetoPostal->getServicoDePostagem()->getCodigo());
        $writer->writeElement('cubagem', (float)$objetoPostal->getCubagem());
        $writer->writeElement('peso', (float)$objetoPostal->getPeso() * 1000);
        $writer->writeElement('rt1');
        $writer->writeElement('rt2');
        $this->writeDestinatario($writer, $objetoPostal->getDestinatario());
        $this->writeDestino($writer, $objetoPostal->getDestino());
        $this->writeServicoAdicional($writer, (array)$objetoPostal->getServicosAdicionais(),$objetoPostal->getServicoDePostagem()->getCodigo());
        $this->writeDimensaoObjeto($writer, $objetoPostal->getDimensao());
        $writer->writeElement('data_postagem_sara');
        $writer->writeElement('status_processamento', 0);
        $writer->writeElement('numero_comprovante_postagem');
        $writer->writeElement('valor_cobrado');
        $writer->endElement();
    }

    private function _($str, $maxLength, $cdata = true, $trim = true)
    {
        if ($str === null) {
            return $str;
        }
        if ($trim) {
            $str = trim($str);
        }
        if ($maxLength) {
            $str = mb_substr($str, 0, $maxLength, 'UTF-8');
        }

        return $str;
    }

    private function writeDestinatario(\XMLWriter $writer, Destinatario $destinatario)
    {
        $writer->startElement('destinatario');
        $writer->startElement('nome_destinatario');
        $writer->writeCdata($this->_($destinatario->getNome(), 50));
        $writer->endElement();
        $writer->startElement('telefone_destinatario');
        $writer->writeCdata($this->_(preg_replace('/[^\d]/', '', $destinatario->getTelefone()), 12));
        $writer->endElement();
        $writer->startElement('celular_destinatario');
        $writer->writeCdata($this->_(preg_replace('/[^\d]/', '', $destinatario->getCelular()), 12));
        $writer->endElement();
        $writer->startElement('email_destinatario');
        $writer->writeCdata($this->_($destinatario->getEmail(), 50));
        $writer->endElement();
        $writer->startElement('logradouro_destinatario');
        $writer->writeCdata($this->_($destinatario->getLogradouro(), 50));
        $writer->endElement();
        $writer->startElement('complemento_destinatario');
        $writer->writeCdata($this->_($destinatario->getComplemento(), 30));
        $writer->endElement();
        $writer->startElement('numero_end_destinatario');
        $writer->writeCdata($this->_($destinatario->getNumero(), 6));
        $writer->endElement();
        $writer->endElement();
    }

    private function writeDestino(\XMLWriter $writer, Destino $destino)
    {
        if ($destino instanceof DestinoNacional) {
            $writer->startElement('nacional');
            $writer->startElement('bairro_destinatario');
            $writer->writeCdata($this->_($destino->getBairro(), 30));
            $writer->endElement();
            $writer->startElement('cidade_destinatario');
            $writer->writeCdata($this->_($destino->getCidade(), 30));
            $writer->endElement();
            $writer->writeElement('uf_destinatario', $this->_($destino->getUf(), 2, false));
            $writer->startElement('cep_destinatario');
            $writer->writeCdata($this->_(preg_replace('/[^\d]/', '', $destino->getCep()), 8));
            $writer->endElement();
            $writer->writeElement('codigo_usuario_postal');
            $writer->writeElement('centro_custo_cliente');
            $writer->writeElement('numero_nota_fiscal', $destino->getNumeroNotaFiscal());
            $writer->writeElement('serie_nota_fiscal', $this->_($destino->getSerieNotaFiscal(), 20));
            $writer->writeElement('valor_nota_fiscal', $destino->getValorNotaFiscal());
            $writer->writeElement('natureza_nota_fiscal', $this->_($destino->getNaturezaNotaFiscal(), 20));
            $writer->startElement('descricao_objeto');
            $writer->writeCdata($this->_($destino->getDescricaoObjeto(), 20));
            $writer->endElement();
            $writer->writeElement('valor_a_cobrar', (float)$destino->getValorACobrar());
            $writer->endElement();
        } else {
            if ($destino instanceof DestinoInternacional) {
                $writer->startElement('internacional');
                $writer->endElement();
            }
        }
    }

    /**
     * @param \XMLWriter $writer
     * @param ServicoAdicional[] $servicosAdicionais
     */
    private function writeServicoAdicional(\XMLWriter $writer, array $servicosAdicionais,$cod_servico_correios)
    {
		//determina o codigo de valor declarado baseado no servico se pac ou sedex
		switch ($cod_servico_correios) {
			case ServicoDePostagem::SERVICE_PAC_41068:
			case ServicoDePostagem::SERVICE_PAC_04510:
			case ServicoDePostagem::SERVICE_PAC_GRANDES_FORMATOS:
			case ServicoDePostagem::SERVICE_PAC_CONTRATO_GRANDES_FORMATOS;
			case ServicoDePostagem::SERVICE_PAC_CONTRATO_UO:
			case ServicoDePostagem::SERVICE_PAC_PAGAMENTO_NA_ENTREGA:
			case ServicoDePostagem::SERVICE_PAC_CONTRATO_AGENCIA:
			case ServicoDePostagem::SERVICE_PAC_REVERSO_CONTRATO_AGENCIA:
			case ServicoDePostagem::SERVICE_PAC_CONTRATO_GRANDES_FORMATOS_LM:
			case ServicoDePostagem::SERVICE_PAC_CONTRATO_AGENCIA_LM:
			case ServicoDePostagem::SERVICE_PAC_REVERSO_LM:
			case ServicoDePostagem::SERVICE_PAC_CONTRATO_UO_LM:
			case ServicoDePostagem::SERVICE_PAC_CONTRATO_AGENCIA_PAGAMENTO_NA_ENTREGA_LM:
			case ServicoDePostagem::SERVICE_PAC_CONTRATO_AGENCIA_TA:
			case ServicoDePostagem::SERVICE_03298:
			case ServicoDePostagem::SERVICE_03085:
			case ServicoDePostagem::SERVICE_03328:
				$cod_declaracao_conteudo = ServicoAdicional::SERVICE_VALOR_DECLARADO_PAC;
				break;
				//mini envios
			case ServicoDePostagem::SERVICE_04227:
			case ServicoDePostagem::SERVICE_04235:
			case ServicoDePostagem::SERVICE_04391:
				$cod_declaracao_conteudo = ServicoAdicional::SERVICE_VALOR_DECLARADO_PAC_MINI;
				break;
				//cartas
			case ServicoDePostagem::SERVICE_CARTA_COMERCIAL_A_FATURAR:
			case ServicoDePostagem::SERVICE_CARTA_REGISTRADA:
			case ServicoDePostagem::SERVICE_CARTA_COM_A_FATURAR_SELO_E_SE:
			case ServicoDePostagem::SERVICE_CARTA_COMERCIAL_REGISTRADA_CTR_EP_MAQ_FRAN:
			case ServicoDePostagem::SERVICE_80292:
			case ServicoDePostagem::SERVICE_80276:
			case ServicoDePostagem::SERVICE_80284:
			case ServicoDePostagem::SERVICE_80250:
			case ServicoDePostagem::SERVICE_80160:
			case ServicoDePostagem::SERVICE_80152:
			case ServicoDePostagem::SERVICE_80357:
			case ServicoDePostagem::SERVICE_80365:
			case ServicoDePostagem::SERVICE_80373:
			case ServicoDePostagem::SERVICE_80381:
			case ServicoDePostagem::SERVICE_80390:
			case ServicoDePostagem::SERVICE_80403:
			case ServicoDePostagem::SERVICE_80411:
			case ServicoDePostagem::SERVICE_80420:
			case ServicoDePostagem::SERVICE_80446:
			case ServicoDePostagem::SERVICE_80454:
			case ServicoDePostagem::SERVICE_80470:
			case ServicoDePostagem::SERVICE_80489:
			case ServicoDePostagem::SERVICE_80497:
			case ServicoDePostagem::SERVICE_80500:
			case ServicoDePostagem::SERVICE_80519:
			case ServicoDePostagem::SERVICE_80527:
			case ServicoDePostagem::SERVICE_80543:
			case ServicoDePostagem::SERVICE_80560:
			case ServicoDePostagem::SERVICE_80586:
			case ServicoDePostagem::SERVICE_80640:
			case ServicoDePostagem::SERVICE_80659:
			case ServicoDePostagem::SERVICE_80675:
			case ServicoDePostagem::SERVICE_80691:
			case ServicoDePostagem::SERVICE_80705:
			//case ServicoDePostagem::SERVICE_80713:
			//case ServicoDePostagem::SERVICE_80721:
			//case ServicoDePostagem::SERVICE_80730:
			//case ServicoDePostagem::SERVICE_80748:
			//case ServicoDePostagem::SERVICE_80756:
			//case ServicoDePostagem::SERVICE_80764:
			//case ServicoDePostagem::SERVICE_80772:
			//case ServicoDePostagem::SERVICE_80780:
				$cod_declaracao_conteudo = ServicoAdicional::SERVICE_VALOR_DECLARADO_CARTA;
				break;
			default:
				$cod_declaracao_conteudo = ServicoAdicional::SERVICE_VALOR_DECLARADO_SEDEX;
				break;
		}
		
        $writer->startElement('servico_adicional');

        // De acordo com o manual este serviço é obrigatório 
        $writer->writeElement('codigo_servico_adicional', ServicoAdicional::SERVICE_REGISTRO);

        foreach ($servicosAdicionais as $servicoAdicional) {
            if ($servicoAdicional->getCodigoServicoAdicional() != ServicoAdicional::SERVICE_REGISTRO) {
                if ($servicoAdicional->getCodigoServicoAdicional() == ServicoAdicional::SERVICE_VALOR_DECLARADO_SEDEX || $servicoAdicional->getCodigoServicoAdicional() == ServicoAdicional::SERVICE_VALOR_DECLARADO_PAC || $servicoAdicional->getCodigoServicoAdicional() == ServicoAdicional::SERVICE_VALOR_DECLARADO_PAC_MINI || $servicoAdicional->getCodigoServicoAdicional() == ServicoAdicional::SERVICE_VALOR_DECLARADO_CARTA) {
					$writer->writeElement('codigo_servico_adicional', $cod_declaracao_conteudo);
                    $writer->writeElement('valor_declarado', (float)$servicoAdicional->getValorDeclarado());
                }else{
					$writer->writeElement('codigo_servico_adicional', $servicoAdicional->getCodigoServicoAdicional());
				}
            }
        }
        $writer->endElement();
    }

    private function writeDimensaoObjeto(\XMLWriter $writer, Dimensao $dimensao)
    {
        $writer->startElement('dimensao_objeto');
        $writer->writeElement('tipo_objeto', $dimensao->getTipo());
        $writer->writeElement('dimensao_altura', $dimensao->getAltura());
        $writer->writeElement('dimensao_largura', $dimensao->getLargura());
        $writer->writeElement('dimensao_comprimento', $dimensao->getComprimento());
        if (!$dimensao->getDiametro()) {
            $writer->writeElement('dimensao_diametro', 0);
        } else {
            $writer->writeElement('dimensao_diametro', $dimensao->getDiametro());
        }
        $writer->endElement();
    }
}
